<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
/**
     * Display all products with add form
     */
    public function product()
    {
        // Get all products with category and images, ordered by newest first
        $products = Product::with(['category', 'images'])
            ->latest()
            ->get();

        // Get all active categories for dropdown
        $categories = Category::orderBy('name')->get();

        return view('product', compact('products', 'categories'));
    }

    /**
     * Store a newly created product
     */
    public function store(Request $request)
{
    // Validate the incoming data
    $validated = $request->validate([
        'category_id' => 'required|exists:categories,id',
        'name' => 'required|string|max:255',
        'price' => 'required|numeric|min:0',
        'discount_price' => 'nullable|numeric|min:0|lt:price',
        'images' => 'required|array|min:1',
        'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
    ]);

    // Auto-generate slug from name
    $slug = Str::slug($validated['name']);

    // Check if slug exists and make it unique
    $count = 1;
    $originalSlug = $slug;
    while (Product::where('slug', $slug)->exists()) {
        $slug = $originalSlug . '-' . $count;
        $count++;
    }

    $validated['slug'] = $slug;

    // Remove 'images' from validated data before creating product
    unset($validated['images']);

    // Create the product
    $product = Product::create($validated);

    // Handle multiple image uploads
    if ($request->hasFile('images')) {
        foreach ($request->file('images') as $image) {
            $imagePath = $image->store('products', 'public');

            ProductImage::create([
                'product_id' => $product->id,
                'image' => $imagePath,
            ]);
        }
    }

    return redirect()
        ->route('admin.products.index')
        ->with('success', 'Product created successfully!');
}

    /**
     * Show edit form for a specific product
     */
    public function edit(Product $product)
    {
        // Load product with images
        $product->load('images');

        // Get all products for the table
        $products = Product::with(['category', 'images'])->latest()->get();

        // Get all categories for dropdown
        $categories = Category::orderBy('name')->get();

        // Return same view but with product to edit
        return view('products.index', compact('products', 'categories', 'product'));
    }

    /**
     * Update the specified product
     */
    public function update(Request $request, Product $product)
    {
        // Validate - exclude current product from unique slug check
        $validated = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lt:price',
            'images' => 'nullable|array', // Images optional on update
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Auto-generate slug from name
        $validated['slug'] = Str::slug($validated['name']);

        // Remove 'images' from validated data before updating product
        unset($validated['images']);

        // Update the product
        $product->update($validated);

        // Handle new image uploads (if any)
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                // Store new image
                $imagePath = $image->store('products', 'public');

                // Save to product_images table
                ProductImage::create([
                    'product_id' => $product->id,
                    'image' => $imagePath,
                ]);
            }
        }

        // Redirect back with success message
        return redirect()
            ->route('products.index')
            ->with('success', 'Product updated successfully!');
    }

    /**
     * Delete the specified product
     * This will also delete all images (cascade)
     */
    public function destroy(Product $product)
    {
        // Delete all product images from storage
        foreach ($product->images as $image) {
            Storage::disk('public')->delete($image->image);
        }

        // Delete the product (images will be deleted automatically due to cascade)
        $product->delete();

        return redirect()
            ->route('products.index')
            ->with('success', 'Product deleted successfully!');
    }

    /**
     * Delete a single image from a product
     * AJAX endpoint
     */
    public function deleteImage(ProductImage $image)
    {
        // Check if this is the last image
        $product = $image->product;
        if ($product->images()->count() <= 1) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete the last image. Product must have at least one image.'
            ], 400);
        }

        // Delete image from storage
        Storage::disk('public')->delete($image->image);

        // Delete image record
        $image->delete();

        return response()->json([
            'success' => true,
            'message' => 'Image deleted successfully!'
        ]);
    }

    /**
 * Get product images for AJAX requests
 */
public function getImages($id)
{
    $product = Product::with('images')->findOrFail($id);
    return response()->json($product->images);
}

    public function show($slug)
{
    $product = Product::where('slug', $slug)
        ->with(['category', 'images'])
        ->firstOrFail();

    // Get related products from same category
    $relatedProducts = Product::where('category_id', $product->category_id)
        ->where('id', '!=', $product->id)
        ->with(['category', 'images'])
        ->take(4)
        ->get();

    return view('product-details', compact('product', 'relatedProducts'));
}
}
