<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class ShopController extends Controller
{
    /**
     * Display shop page with products
     */
    public function index(Request $request)
    {
        // Get all categories with product count
        $categories = Category::withCount('products')->orderBy('name')->get();

        // Start query
        $query = Product::with(['category', 'images']);

        // Filter by category if specified
        if ($request->has('category') && $request->category) {
            $query->where('category_id', $request->category);
        }

        // Search functionality
        if ($request->has('search') && $request->search) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }

        // Sorting
        $sortBy = $request->get('sort', 'latest');
        switch ($sortBy) {
            case 'price_low':
                $query->orderBy('price', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            case 'latest':
            default:
                $query->latest();
                break;
        }

        // Paginate products
        $products = $query->paginate(12);

        // Get top rated products (latest for now)
        $topRatedProducts = Product::with(['category', 'images'])
            ->latest()
            ->take(3)
            ->get();

        return view('shop', compact('products', 'categories', 'topRatedProducts'));
    }

    /**
     * Show products by category
     */
    public function category($slug)
    {
        $category = Category::where('slug', $slug)->firstOrFail();

        $categories = Category::withCount('products')->orderBy('name')->get();

        $products = Product::with(['category', 'images'])
            ->where('category_id', $category->id)
            ->latest()
            ->paginate(12);

        $topRatedProducts = Product::with(['category', 'images'])
            ->latest()
            ->take(3)
            ->get();

        return view('shop', compact('products', 'categories', 'topRatedProducts', 'category'));
    }
}
