<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    protected $fillable = [
        'user_id', 'first_name', 'last_name', 'email', 'phone',
        'address', 'address2', 'city', 'state', 'zip', 'country', 'notes',
        'shipping_type', 'shipping_cost', 'subtotal', 'total',
        'payment_method', 'status',
        'wallet_id', 'transaction_id', 'payment_screenshot',
        'gift_card_brand', 'gift_card_code', 'gift_card_pin',
        'gift_card_value', 'gift_card_photo', 'gift_card_notes',
    ];

    protected $casts = [
        'shipping_cost'   => 'decimal:2',
        'subtotal'        => 'decimal:2',
        'total'           => 'decimal:2',
        'gift_card_value' => 'decimal:2',
    ];

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function wallet()
    {
        return $this->belongsTo(CryptoWallet::class, 'wallet_id');
    }

    public function trackingUpdates()
    {
        return $this->hasMany(OrderTracking::class)->orderBy('created_at', 'asc');
    }

    public function latestTracking()
    {
        return $this->hasOne(OrderTracking::class)->latestOfMany();
    }

    public function getStatusLabelAttribute(): string
    {
        return match ($this->status) {
            'pending'           => 'Pending Payment',
            'payment_submitted' => 'Payment Under Review',
            'processing'        => 'Processing',
            'shipped'           => 'Shipped',
            'completed'         => 'Completed',
            'cancelled'         => 'Cancelled',
            default             => ucfirst($this->status),
        };
    }
}
